library(haven)
library(tidyverse)
library(car)
library(psych)
library(GPArotation)
library(sjmisc)
library(sjPlot)
library(lsr)
library(lavaan)
library(semTools)
library(jtools)
library(lmtest)
library(sandwich)
library(margins)
library(ggplot2)
library(gridExtra)
library(mediation)


# 1 - Data preparation:

# Read raw dataset:

data.raw <- read_sav("Who, If Not Science, Can You Trust to Guide You Through a Crisis_JCOM_Data.sav")

# Sort out incomplete cases from dataset:

data.initial <- data.raw %>% filter(QC04_01 == 0)


# Recode variables:

data.edited <- data.initial

# sociodemographics:
data.edited$gender.female <- car::recode(data.initial$SD01, "1 = 1; 2 = 0; NA = NA")
data.edited$education <- car::recode(data.initial$SD03, "1 = 1; 2 = 1; 3 = 1; 4 = 2; 5 = 2; NA = NA; else = 3")
data.edited$education.low <- car::recode(data.initial$SD03, "1 = 1; 2 = 1; 3 = 1; NA = NA; else = 0")
data.edited$education.medium <- car::recode(data.initial$SD03, "4 = 1; 5 = 1; NA = NA; else = 0")
data.edited$residency.east <- car::recode(data.initial$SD04, "c(4, 8, 13, 14, 16) = 1; NA = NA; else = 0")

# occupational involvement with science:
data.edited$involvement <- ifelse(data.initial$SD06_01 == 2 |data.initial$SD06_02 == 2,
                                  1,                         
                                  0)                         

# populist partisanship (AfD):
data.edited$partisanship.AfD <- car::recode(data.initial$PD02, "1 = 1; NA = NA; else = 0")

# alternative media outlets:
data.edited$achgut <- car::recode(data.initial$MN03_01, "-1 = 1; NA = NA")
data.edited$compact <- car::recode(data.initial$MN03_02, "-1 = 1; NA = NA")
data.edited$epoch <- car::recode(data.initial$MN03_03, "-1 = 1; NA = NA")
data.edited$jf <- car::recode(data.initial$MN03_04, "-1 = 1; NA = NA")
data.edited$manova <- car::recode(data.initial$MN03_05, "-1 = 1; NA = NA")
data.edited$nds <- car::recode(data.initial$MN03_06, "-1 = 1; NA = NA")
data.edited$pi <- car::recode(data.initial$MN03_07, "-1 = 1; NA = NA")
data.edited$tichy <- car::recode(data.initial$MN03_08, "-1 = 1; NA = NA")

# Dichotomize (skewed) alternative media use variables (including Telegram use):

data.edited$telegram.dummy <- car::recode(data.edited$MN02_03, "1 = 0; NA = NA; else = 1")
data.edited$achgut.dummy <- car::recode(data.edited$achgut, "1 = 0; NA = NA; else = 1")
data.edited$compact.dummy <- car::recode(data.edited$compact, "1 = 0; NA = NA; else = 1")
data.edited$epoch.dummy <- car::recode(data.edited$epoch, "1 = 0; NA = NA; else = 1")
data.edited$jf.dummy <- car::recode(data.edited$jf, "1 = 0; NA = NA; else = 1")
data.edited$manova.dummy <- car::recode(data.edited$manova, "1 = 0; NA = NA; else = 1")
data.edited$nds.dummy <- car::recode(data.edited$nds, "1 = 0; NA = NA; else = 1")
data.edited$pi.dummy <- car::recode(data.edited$pi, "1 = 0; NA = NA; else = 1")
data.edited$tichy.dummy <- car::recode(data.edited$tichy, "1 = 0; NA = NA; else = 1")

# Compute (mean) indices for the scales:

data.edited <- data.edited %>%
  
  # Trust expectations towards science:
  mutate(trust_expectations = rowMeans(data.frame(VE01_01,
                                                  VE01_02,
                                                  VE01_03,
                                                  VE01_04,
                                                  VE01_05,
                                                  VE01_06),
                                   na.rm = TRUE)) %>% 
  # Need for orientation:
  mutate(nfo = rowMeans(data.frame(PM01_01,
                                   PM01_02,
                                   PM01_03,
                                   PM01_04,
                                   PM01_05,
                                   PM01_06),
                        na.rm = TRUE)) %>%
  
  # Established online source use:
  mutate(est_media = rowMeans(data.frame(MN02_01,
                                         MN02_02),
                              na.rm = TRUE)) %>%
  
  # Alternative online media outlet use (dichotomous):
  mutate(alt_media = rowSums(data.frame(achgut.dummy,
                                         compact.dummy,
                                         epoch.dummy,
                                         jf.dummy,
                                         manova.dummy,
                                         nds.dummy,
                                         pi.dummy,
                                         tichy.dummy),
                              na.rm = TRUE)) %>%
  mutate(alt_media.dummy = dplyr::recode(alt_media, '0' = 0, .default = 1)) %>%
  
  # Perceived trustworthiness of scientists:
  mutate(trustworthiness_science = rowMeans(data.frame(WV03_01,
                                                       WV03_02,
                                                       WV03_03),
                                            na.rm = TRUE)) %>%
  
  # Populist attitudes:
  mutate(pop_attitudes = rowMeans(data.frame(PD03_01,
                                             PD03_02,
                                             PD03_03),
                                  na.rm = TRUE)) %>%
  
  # Institutional trust:
  mutate(inst_trust = rowMeans(data.frame(WV02_01,
                                          WV02_02,
                                          WV02_03),
                                  na.rm = TRUE)) 


# Vectorizing all focal variables:

data.edited$gender.female <- as.numeric(data.edited$gender.female)
data.edited$age <- as.numeric(data.edited$SD02)
data.edited$education <- as.numeric(data.edited$education)
data.edited$education.low <- as.numeric(data.edited$education.low)
data.edited$education.medium <- as.numeric(data.edited$education.medium)
data.edited$residency.east <- as.numeric(data.edited$residency.east)
data.edited$involvement <- as.numeric(data.edited$involvement)
data.edited$interest <- as.numeric(data.edited$PM04)
data.edited$trustworthiness_science <- as.numeric(data.edited$trustworthiness_science)
data.edited$ideology <- as.numeric(data.edited$PD01)
data.edited$partisanship.AfD <- as.numeric(data.edited$partisanship.AfD)
data.edited$pop_attitudes <- as.numeric(data.edited$pop_attitudes)
data.edited$inst_trust <- as.numeric(data.edited$inst_trust)
data.edited$trust_expectations <- as.numeric(data.edited$trust_expectations)
data.edited$nfo <- as.numeric(data.edited$nfo)
data.edited$est_media <- as.numeric(data.edited$est_media)
data.edited$alt_media <- as.numeric(data.edited$alt_media)
data.edited$alt_media.dummy <- as.numeric(data.edited$alt_media.dummy)
data.edited$telegram <- as.numeric(data.edited$MN02_03)
data.edited$telegram.dummy <- as.numeric(data.edited$telegram.dummy)


# Mean centering all continuous variables:

data.edited <- data.edited %>%
  mutate(age.c = c(scale(age, scale = FALSE))) %>% 
  mutate(interest.c = c(scale(interest, scale = FALSE))) %>%
  mutate(ideology.c = c(scale(ideology, scale = FALSE))) %>% 
  mutate(pop_attitudes.c = c(scale(pop_attitudes, scale = FALSE))) %>%
  mutate(inst_trust.c = c(scale(inst_trust, scale = FALSE))) %>%
  mutate(trustworthiness_science.c = c(scale(trustworthiness_science, scale = FALSE))) %>% 
  mutate(nfo.c = c(scale(nfo, scale = FALSE))) %>% 
  mutate(trust_expectations.c = c(scale(trust_expectations, scale = FALSE))) %>% 
  mutate(est_media.c = c(scale(est_media, scale = FALSE))) %>% 
  mutate(alt_media.c = c(scale(alt_media, scale = FALSE))) %>% 
  mutate(telegram.c = c(scale(telegram, scale = FALSE)))


# Create dataset with only valid cases (net sample):

data.net <- data.edited %>%
  filter(ID_mehrfach == 0) %>%         # sort out multiple participants
  filter(TIME_RSI < 2) %>%             # sort out speeders (total questionnaire)
  filter(TIME_RSI_trust.items < 2) %>% # sort out speeders (trust scale items)
  filter(trust.items_pos_irv > 0) %>%  # sort out straight-liners (positive trust items)
  filter(trust.items_neg_irv > 0)      # sort out straight-liners (negative trust items)


# Save new datasets in working directory:

write_sav(data.edited, "JCOM.edited_data.sav")

write_sav(data.net, "JCOM.net_sample.sav")



# 2 - Preliminary analyses:

# Validating "trust expectations towards science" measure by confirmatory factor analysis:

# Measurement model 1:

# 2 latent factors:
#
#   - Perceived trustworthiness of scientists (3 items)
#
#   - Trust expectations towards science (6 items)


# Define CFA:

cfa.2factors <- '

# Latent factor 1: Perceived trustworthiness of scientists

trustworthiness.l =~
WV03_01 +
WV03_02 +
WV03_03

# Latent factor 2: Trust expectations towards science

trust_expectations.l =~
VE01_01 +
VE01_02 +
VE01_03 +
VE01_04 +
VE01_05 +
VE01_06

# Covariance between latent factors:

trust_expectations.l ~~
trustworthiness.l

'

# Fit CFA:

cfa.2factors.fit <- sem(model = cfa.2factors,
                        data = data.net,
                        estimator = "MLR",
                        orthogonal = TRUE)

# Show CFA:

summary(cfa.2factors.fit,
        fit.measures = TRUE,
        standardize = TRUE,
        rsquare = TRUE,
        modindices = FALSE)

# Show standardized estimates with 90% CI:

standardizedSolution(cfa.2factors.fit, level = 0.9)

# Calculate construct reliability of the factors:

compRelSEM(cfa.2factors.fit)

# Calculate average variance extracted (AVE) of the factors:

AVE(cfa.2factors.fit)


# Measurement model 2:

# 1 latent factor:
#
#   - Perceived trustworthiness of scientists + trust expectations towards science (9 items)


# Define CFA:

cfa.1factor <- '

# Latent factor 1: Guidance trust in science

trustworthiness_and_trust_expectations.l =~
WV03_01 +
WV03_02 +
WV03_03 +
VE01_01 +
VE01_02 +
VE01_03 +
VE01_04 +
VE01_05 +
VE01_06
'

# Fit CFA:

cfa.1factor.fit <- sem(model = cfa.1factor,
                       data = data.net,
                       estimator = "MLR",
                       orthogonal = TRUE)

# Show CFA:

summary(cfa.1factor.fit,
        fit.measures = TRUE,
        standardize = TRUE,
        rsquare = TRUE,
        modindices = FALSE)

# Comparing measurement model 1 and 2 through Chi-square difference test:

lavTestLRT(cfa.2factors.fit, cfa.1factor.fit)



# Checking internal consistency of the scales...:

# ...using McDonald's omega for scales with continuous items:

# Trust expectations towards science:

data.net %>% dplyr::select(VE01_01,
                           VE01_02,
                           VE01_03,
                           VE01_04,
                           VE01_05,
                           VE01_06) %>%
  psych::omega(nfactors = 1)


# Need for orientation:

data.net %>% dplyr::select(PM01_01,
                           PM01_02,
                           PM01_03,
                           PM01_04,
                           PM01_05,
                           PM01_06) %>%
  psych::omega(nfactors = 1)


# Perceived trustworthiness of scientists:

data.net %>% dplyr::select(WV03_01,
                           WV03_02,
                           WV03_03) %>%
  psych::omega(nfactors = 1)


# Populist attitudes:

data.net %>% dplyr::select(PD03_01,
                           PD03_02,
                           PD03_03) %>%
  psych::omega(nfactors = 1)


# Institutional trust:

data.net %>% dplyr::select(WV02_01,
                           WV02_02,
                           WV02_03) %>%
  psych::omega(nfactors = 1)


# ... using Kuder-Richardson Formula 20:

# Alternative online media use:

data.net %>% dplyr::select(achgut.dummy,
                           compact.dummy,
                           epoch.dummy,
                           jf.dummy,
                           manova.dummy,
                           nds.dummy,
                           pi.dummy,
                           tichy.dummy) %>%
  psych::alpha() # Cronbach's Alpha is equivalent to the Kuder-Richardson Formula 20 in case of only dichotomous items.

# ... using Spearman-Brown coefficient for scales with only two items:

# Established online source use:
data.net %>% dplyr::select(MN02_01,
                           MN02_02) %>%
  psych::alpha() # Cronbach's Alpha is equivalent to the Spearman-Brown coefficient in case of two items that have the same length.


# Descriptive statistics for all continuous scale items:  

descriptives.continuous_items <- data.net %>% dplyr::select(
  
  # trust expectations items:
  VE01_01,
  VE01_02,
  VE01_03,
  VE01_04,
  VE01_05,
  VE01_06,
  
  # need for orientation items:
  PM01_01,
  PM01_02,
  PM01_03,
  PM01_04,
  PM01_05,
  PM01_06,
  
  # perceived trustworthiness items:
  WV03_01,
  WV03_02,
  WV03_03,
  
  # populist attitudes items:
  PD03_01,
  PD03_02,
  PD03_03,
  
  # institutional trust items:
  WV02_01,
  WV02_02,
  WV02_03,
  
  # established online source use:
  MN02_01,
  MN02_02) %>%
  
  psych::describe()


# Descriptive statistics for all continuous model variables:

descriptives.continuous_variables <- data.net %>% dplyr::select(age,
                                                                interest,
                                                                trustworthiness_science,
                                                                ideology,
                                                                pop_attitudes,
                                                                inst_trust,
                                                                trust_expectations,
                                                                nfo,
                                                                est_media) %>%
  psych::describe()


# Frequency tables for all categorical scale items (alternative media outlet use):

frequencies.categorical_scales <- data.net %>% dplyr::select(achgut.dummy,
                                                             compact.dummy,
                                                             epoch.dummy,
                                                             jf.dummy,
                                                             manova.dummy,
                                                             nds.dummy,
                                                             pi.dummy,
                                                             tichy.dummy) %>%
  frq(out = "viewer")


# Frequency tables for all categorical model variables:

frequencies.categorical_variables <- data.net %>% dplyr::select(gender.female,
                                                                education,
                                                                residency.east,
                                                                involvement,
                                                                partisanship.AfD,
                                                                telegram.dummy,
                                                                alt_media.dummy) %>%
  frq(out = "viewer")


# Comparing the sociodemographic variable distribution of the net sample (N = 1,223) against
# the values of the initial sample (N = 1,419)...:

# ... using 95%-confidence intervals for continuous variables:

# Age:

lsr::ciMean(data.edited$age, na.rm = TRUE) # Initial sample 95% CI 

lsr::ciMean(data.net$age, na.rm = TRUE) # Net sample 95% CI


# ... using Chi-square test for given probabilities for categorical variables:

# Initial sampling distribution of gender, education, and residency:

data.edited %>% dplyr::select(gender.female, education, residency.east) %>%
  frq(out = "viewer")

# Chi-square test gender:
chisq.test(table(data.net$gender.female), p = c(0.496, 0.504))

# Chi-square test education:
chisq.test(table(data.net$education), p = c(0.33, 0.40, 0.27))

# Chi-square test residency:
chisq.test(table(data.net$residency.east), p = c(0.85, 0.15))



# 3 - Hypothesis testing:

# Select focal variables from dataset:

data.testing <- data.net %>% dplyr::select(gender.female,
                                           age.c,
                                           education,
                                           education.low,
                                           education.medium,
                                           residency.east,
                                           involvement,
                                           interest.c,
                                           trustworthiness_science.c,
                                           ideology.c,
                                           partisanship.AfD,
                                           pop_attitudes.c,
                                           inst_trust.c,
                                           trust_expectations,
                                           trust_expectations.c,
                                           nfo.c,
                                           est_media.c,
                                           alt_media.c,
                                           alt_media.dummy,
                                           telegram.c,
                                           telegram.dummy) %>% 
  
  # Listwise deletion of missing values on any focal variable:
  drop_na()

# Hypothesis H1

# Multiple linear Regression

# Model 1:

# DV: Trust expectations towards science

# IV: Perceived trustworthiness of scientists

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Need for orientation,
# Institutional trust

model.1 <- lm(trust_expectations.c ~
                gender.female +
                age.c +
                education.low +
                education.medium +
                residency.east +
                involvement +
                interest.c +
                ideology.c +
                partisanship.AfD +
                pop_attitudes.c +
                inst_trust.c +
                nfo.c +
                trustworthiness_science.c,
              data = data.testing)

# Show regression results:
summ(model.1)

# Regression table with robust standard errors and 95% confidence intervals:

model.1.tab <- tab_model(model.1,
                         vcov.fun = "HC3",
                         show.se = TRUE,
                         show.std = TRUE,
                         show.stat = TRUE,
                         file = "model.1.html")


# Hypothesis H2

# Multiple linear Regression

# Model 2a:

# DV: Established online source use

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Telegram use,
# Alternative media outlets use,
# Need for orientation
# Perceived trustworthiness of scientists,

model.2a <- lm(est_media.c ~
                 gender.female +
                 age.c +
                 education.low +
                 education.medium +
                 residency.east+
                 involvement +
                 interest.c +
                 ideology.c +
                 partisanship.AfD +
                 pop_attitudes.c +
                 inst_trust.c +
                 telegram.dummy +
                 alt_media.dummy +
                 nfo.c +
                 trustworthiness_science.c,
               data = data.testing)

# Show regression results:
summ(model.2a)

# Regression table with robust standard errors and 95% confidence intervals:

model.2a.tab <- tab_model(model.2a,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.std = TRUE,
                          show.stat = TRUE,
                          file = "model.2a.html")


# Multiple linear Regression

# Model 2b:

# DV: Established online source use

# IV: Trust expectations towards science

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Telegram use,
# Alternative media outlets use,
# Need for orientation
# Perceived trustworthiness of scientists

model.2b <- lm(est_media.c ~
                 gender.female +
                 age.c +
                 education.low +
                 education.medium +
                 residency.east+
                 involvement +
                 interest.c +
                 ideology.c +
                 partisanship.AfD +
                 pop_attitudes.c +
                 inst_trust.c +
                 telegram.dummy +
                 alt_media.dummy +
                 nfo.c +
                 trustworthiness_science.c +
                 trust_expectations.c,
               data = data.testing)

# Show regression results:
summ(model.2b)

# Regression table with robust standard errors and 95% confidence intervals:

model.2b.tab <- tab_model(model.2b,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.std = TRUE,
                          show.stat = TRUE,
                          file = "model.2b.html")

# Calculate Delta R-squared:
summary(model.2b)$r.squared - summary(model.2a)$r.squared

# Compare model fit of model 1a and 1b through ANOVA:
anova(model.2a, model.2b)


# Hypothesis H3a

# Binary logistic regression
# Model 3a:

# DV: Telegram use

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Need for orientation
# Perceived trustworthiness of scientists,

model.3a <- glm(telegram.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  nfo.c+
                  trustworthiness_science.c,
                family = binomial,
                data = data.testing)

# Show regression results:
summ(model.3a)

# Regression table with robust standard errors and 95% confidence intervals:

model.3a.tab <- tab_model(model.3a,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.3a.html")


# Binary logistic regression

# Model 3b:

# DV: Telegram use

# IV: Trust expectations towards science

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Need for orientation,
# Perceived trustworthiness of scientists

model.3b <- glm(telegram.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  nfo.c +
                  trustworthiness_science.c +
                  trust_expectations.c,
                family = binomial,
                data = data.testing)

# Show regression results:

summ(model.3b)

# Regression table with robust standard errors and 95% confidence intervals:

model.3b.tab <- tab_model(model.3b,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.3b.html")

# Compare model fit of model 2a and 2b through Chi-square test:

models.chi <- model.3a$deviance - model.3b$deviance
chi.df <- model.3a$df.residual - model.3b$df.residual
chi.p <- 1 - pchisq(models.chi, chi.df)


# Calculate average marginal effects:

AME.3b <- margins(model.3b, vcov = vcovHC(model.3b))

summary(AME.3b)

# Plot effect of guidance trust on predicted probabilities of Telegram use:

AME.3b.plot <-  plot_model(model.3b, type = "pred",
                           terms = "trust_expectations.c [all]",
                           vcov.fun = "HC3",
                           auto.label = FALSE,
                           title = "",
                           axis.title = c("Trust expectations towards science", "Predicted probability of using Telegram"),
                           colors = "red") +
  scale_x_continuous(breaks = seq(-3, 1, by = 1),   # Values of the centered variable (-3 to 1)
                     labels = seq(1, 5, by = 1)) +  # Values of the uncentered variable (1 to 5)
  theme_apa()


# Hypothesis H3b

# Binary logistic regression

# Model 3c:

# DV: Alternative media outlet use

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Need for orientation,
# Perceived trustworthiness of scientists

model.3c <- glm(alt_media.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  nfo.c+
                  trustworthiness_science.c,
                family = binomial,
                data = data.testing)

# Show regression results:

summ(model.3c)

# Regression table with robust standard errors and 95% confidence intervals:

model.3c.tab <- tab_model(model.3c,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.3c.html")


# Binary logistic regression

# Model 3d:

# DV: Alternative media outlet use

# IV: Trust expectations towards science

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Need for orientation,
# Perceived trustworthiness of scientists

model.3d <- glm(alt_media.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  nfo.c +
                  trustworthiness_science.c +
                  trust_expectations.c,
                family = binomial,
                data = data.testing)

# Show regression results:
summ(model.3d)

# Regression table with robust standard errors and 95% confidence intervals:

model.3d.tab <- tab_model(model.3d,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.3d.html")

# Compare model fit of model 2c and 2d through Chi-square test:

models.chi <- model.3c$deviance - model.3d$deviance
chi.df <- model.3c$df.residual - model.3d$df.residual
chi.p <- 1 - pchisq(models.chi, chi.df)


# Calculate average marginal effects

AME.3d <- margins(model.3d, vcov = vcovHC(model.3d))

summary(AME.3d)


# Plot effect of guidance trust on predicted probabilities of alternative media outlet use:

AME.3d.plot <- plot_model(model.3d, type = "pred",
                          terms = "trust_expectations.c [all]",
                          vcov.fun = "HC3",
                          auto.label = FALSE,
                          title = "",
                          axis.title = c("Trust expectations towards science", "Predicted probability of using alternative media"),
                          colors = "blue") +
  scale_x_continuous(breaks = seq(-3, 1, by = 1),   # Values of the centered variable (-3 to 1)
                     labels = seq(1, 5, by = 1)) +  # Values of the uncentered variable (1 to 5)
  theme_apa()


# Combine the two marginal effects plots into one figure:

AME.plots <- grid.arrange(AME.3b.plot, AME.3d.plot, nrow = 1)


# Hypothesis H4a

# Multiple linear Regression
# Model 4a:
# DV: Established online source use

# IV: Trust expectations towards science

# Moderator: Need for orientation

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Epistemic Trustworthiness of science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Telegram use,
# Alternative media outlets use

model.4a <- lm(est_media.c ~
                 gender.female +
                 age.c +
                 education.low +
                 education.medium +
                 residency.east+
                 involvement +
                 interest.c +
                 ideology.c +
                 partisanship.AfD +
                 pop_attitudes.c +
                 inst_trust.c +
                 telegram.dummy +
                 alt_media.dummy +
                 trustworthiness_science.c +
                 nfo.c*
                 trust_expectations.c,
               data = data.testing)

# Show regression results:
summ(model.4a)

# Regression table with robust standard errors and 95% confidence intervals:

model.4a.tab <- tab_model(model.4a,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.std = TRUE,
                          show.stat = TRUE,
                          file = "model.4a.html")

# Calculate Delta R-squared:
summary(model.4a)$r.squared - summary(model.2b)$r.squared

# Compare model fit of model 3a and 1b through ANOVA:
anova(model.2b, model.4a)


# Hypothesis H4b

# Binary logistic regression

# Model 4b:

# DV: Telegram use

# IV: Trust expectations towards science

# Moderator: Need for orientation

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Perceived trustworthiness of scientists

model.4b <- glm(telegram.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  trustworthiness_science.c +
                  nfo.c*
                  trust_expectations.c,
                family = binomial,
                data = data.testing)

# Show regression results:

summ(model.4b)

# Regression table with robust standard errors and 95% confidence intervals:

model.4b.tab <- tab_model(model.4b,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.4b.html")

# Compare model fit of model 4b and 3b through Chi-square test:

models.chi <- model.3b$deviance - model.4b$deviance
chi.df <- model.3b$df.residual - model.4b$df.residual
chi.p <- 1 - pchisq(models.chi, chi.df)


# Hypothesis H4c

# Binary logistic regression

# Model 4c:

# DV: Alternative media outlet use

# IV: Trust expectations towards science

# Moderator: Need for orientation

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Perceived trustworthiness of scientists

model.4c <- glm(alt_media.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  trustworthiness_science.c +
                  nfo.c*
                  trust_expectations.c,
                family = binomial,
                data = data.testing)

# Show regression results:
summ(model.4c)

# Regression table with robust standard errors and 95% confidence intervals:

model.4c.tab <- tab_model(model.4c,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.4c.html")

# Compare model fit of model 4c and 3d through Chi-square test:

models.chi <- model.3d$deviance - model.4c$deviance
chi.df <- model.3d$df.residual - model.4c$df.residual
chi.p <- 1 - pchisq(models.chi, chi.df)


# Causal mediation analyses:

# Causal mediation analysis 1:

# IV: Perceived trustworthiness of scientists
# M: Trust expectations towards science
# DV: Established online source use

set.seed(58973)
mediation.1 <- mediate(model.1,  # Path a: Perceived trustworthiness of scientists -> Trust expectations towards science (model 1)
                       model.2b, # Path b: Trust expectations towards science -> established online source use (model 2b)
                       treat = "trustworthiness_science.c", # IV: Perceived trustworthiness of scientists
                       mediator = "trust_expectations.c",   # M: Trust expectations towards science
                       treat.value = 1,      # Mean + 1
                       control.value = 0,    # Mean
                       boot = TRUE,          # use nonparametric bootstrap to estimate 95% confidence intervals
                       boot.ci.type = "bca", # use bias-corrected and accelerated bootstrapping
                       sims = 10000)         # number of Monte Carlo draws for nonparametric bootstrap = 10,000

# Causal mediation analysis 2:

# IV: Perceived trustworthiness of scientists
# M: Trust expectations towards science
# DV: Telegram use

set.seed(27690)
mediation.2 <- mediate(model.1,  # Path a: Perceived trustworthiness of scientists -> Trust expectations towards science (model 1)
                       model.3b, # Path b: Trust expectations towards science -> Telegram use (model 3b)
                       treat = "trustworthiness_science.c", # IV: Perceived trustworthiness of scientists
                       mediator = "trust_expectations.c",   # M: Trust expectations towards science
                       treat.value = 1,      # Mean + 1
                       control.value = 0,    # Mean
                       boot = TRUE,          # use nonparametric bootstrap to estimate 95% confidence intervals
                       boot.ci.type = "bca", # use bias-corrected and accelerated bootstrapping
                       sims = 10000)         # number of Monte Carlo draws for nonparametric bootstrap = 10,000

# Causal mediation analysis 3:

# IV: Perceived trustworthiness of scientists
# M: Trust expectations towards science
# DV: Alternative media outlet use

set.seed(37104)
mediation.3 <- mediate(model.1,  # Path a: Perceived trustworthiness of scientists -> Trust expectations towards science (model 1)
                       model.3d, # Path b: Trust expectations towards science -> Alternative media outlet use (model 3d)
                       treat = "trustworthiness_science.c", # IV: Perceived trustworthiness of scientists
                       mediator = "trust_expectations.c",   # M: Trust expectations towards science
                       treat.value = 1,      # Mean + 1
                       control.value = 0,    # Mean
                       boot = TRUE,          # use nonparametric bootstrap to estimate 95% confidence intervals
                       boot.ci.type = "bca", # use bias-corrected and accelerated bootstrapping
                       sims = 10000)         # number of Monte Carlo draws for nonparametric bootstrap = 10,000


# Show results of causal mediation analyses:
summary(mediation.1)
summary(mediation.2)
summary(mediation.3)



# Sensitivity analysis for binary logistic regressions with "alternative media
# outlet use" as dependent variable:

# Remove all participants who do not know any of the alternative media outlets
# (under-sampling):

data.robust <- data.net %>% filter(MN03_01 != (-1) |
                                     MN03_02 != (-1) |
                                     MN03_03 != (-1) |
                                     MN03_04 != (-1) |
                                     MN03_05 != (-1) |
                                     MN03_06 != (-1) |
                                     MN03_07 != (-1) |
                                     MN03_08 != (-1))


# Rerun binary logistic regressions for hypothesis testing with reduced dataset:

# Hypothesis H3b:

# Binary logistic regression

# Model 5a:

# DV: Alternative media outlet use

# IV: Trust expectations towards science

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Need for orientation,
# Perceived trustworthiness of scientists

model.5a <- glm(alt_media.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  nfo.c +
                  trustworthiness_science.c +
                  trust_expectations.c,
                family = binomial,
                data = data.robust)

# Show regression results:
summ(model.5a)

# Regression table with robust standard errors and 95% confidence intervals:

model.5a.tab <- tab_model(model.5a,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.5a.html")


# Hypothesis H4c

# Binary logistic regression

# Model 5b:

# DV: Alternative media outlet use

# IV: Trust expectations towards science

# Moderator: Need for orientation

# Controls:
# Gender,
# Age,
# Education,
# Residency,
# Occupational involvement with science,
# Interest in science,
# Populist partisanship (AfD),
# Populist attitudes,
# Institutional trust,
# Established online source use,
# Perceived trustworthiness of scientists

model.5b <- glm(alt_media.dummy ~
                  gender.female +
                  age.c +
                  education.low +
                  education.medium +
                  residency.east+
                  involvement +
                  interest.c +
                  ideology.c +
                  partisanship.AfD +
                  pop_attitudes.c +
                  inst_trust.c +
                  est_media.c +
                  trustworthiness_science.c +
                  nfo.c*
                  trust_expectations.c,
                family = binomial,
                data = data.robust)

# Show regression results:
summ(model.5b)

# Regression table with robust standard errors and 95% confidence intervals:

model.5b.tab <- tab_model(model.5b,
                          vcov.fun = "HC3",
                          show.se = TRUE,
                          show.stat = TRUE,
                          file = "model.5b.html")